#include <stdint.h>

// The A5 core has both L1 and L2 cache
#define PSP_HAS_L1_CACHE	1
#define PSP_HAS_L2_CACHE	1

#define _STRINGIFY(x) #x
#define MRC(coproc, opcode1, Rt, CRn, CRm, opcode2) \
asm volatile ("mrc " _STRINGIFY(p##coproc) " , " _STRINGIFY(opcode1) \
" , %[output] , " _STRINGIFY(c##CRn) " , " _STRINGIFY(c##CRm) " , " \
_STRINGIFY(opcode2) : [output] "=r" (Rt) : )

#define MCR(coproc, opcode1, Rt, CRn, CRm, opcode2) \
asm volatile ("mcr " _STRINGIFY(p##coproc) " , " _STRINGIFY(opcode1) \
" , %[input] , " _STRINGIFY(c##CRn) " , " _STRINGIFY(c##CRm) " , " \
_STRINGIFY(opcode2) : : [input] "r" (Rt))

#define DSB()   asm("dsb")

typedef volatile uint32_t vuint32_t;
typedef volatile uint16_t vuint16_t;
typedef volatile uint8_t vuint8_t;

typedef struct L2C_MemMap 
{
  uint32_t reg0_cache_id;                          /**< Performance Monitor Control Register, offset: 0x0 */
  uint32_t reg0_cache_type;                        /**< Cache Type Register, offset: 0x4 */
  uint8_t RESERVED_0[248];
  uint32_t reg1_control;                           /**< Control Register, offset: 0x100 */
  uint32_t reg1_aux_control;                       /**< Auxiliary Control Register, offset: 0x104 */
  uint32_t reg1_tag_ram_control;                   /**< Tag RAM Latency Control Register, offset: 0x108 */
  uint32_t reg1_data_ram_control;                  /**< Data RAM Latency Control Register, offset: 0x10C */
  uint8_t RESERVED_1[240];
  uint32_t reg2_ev_counter_ctrl;                   /**< Event Counter Control Register, offset: 0x200 */
  uint32_t reg2_ev_counter1_cfg;                   /**< Event Counter Configuration Register, offset: 0x204 */
  uint32_t reg2_ev_counter0_cfg;                   /**< Event Counter Configuration Register, offset: 0x208 */
  uint32_t reg2_ev_counter1;                       /**< Event counter value register 1, offset: 0x20C */
  uint32_t reg2_ev_counter0;                       /**< Event counter value register 0, offset: 0x210 */
  uint32_t reg2_int_mask;                          /**< Interrupt Mask Register, offset: 0x214 */
  uint32_t reg2_int_mask_status;                   /**< Masked Interrupt Status Register, offset: 0x218 */
  uint32_t reg2_int_raw_status;                    /**< Raw Interrupt Status Register, offset: 0x21C */
  uint32_t reg2_int_clear;                         /**< Interrupt Clear Register, offset: 0x220 */
  uint8_t RESERVED_2[1292];
  uint32_t reg7_cache_sync;                        /**< Cache Sync, offset: 0x730 */
  uint8_t RESERVED_3[60];
  uint32_t reg7_inv_pa;                            /**< Invalidate Line by PA, offset: 0x770 */
  uint8_t RESERVED_4[8];
  uint32_t reg7_inv_way;                           /**< Invalidate by Way, offset: 0x77C */
  uint8_t RESERVED_5[48];
  uint32_t reg7_clean_pa;                          /**< Clean Line by PA, offset: 0x7B0 */
  uint8_t RESERVED_6[4];
  uint32_t reg7_clean_index;                       /**< Clean Line by Set/Way, offset: 0x7B8 */
  uint32_t reg7_clean_way;                         /**< Clean by Way, offset: 0x7BC */
  uint8_t RESERVED_7[48];
  uint32_t reg7_clean_inv_pa;                      /**< Clean and Invalidate Line by PA, offset: 0x7F0 */
  uint8_t RESERVED_8[4];
  uint32_t reg7_clean_inv_index;                   /**< Clean and Invalidate Line by Set/Way, offset: 0x7F8 */
  uint32_t reg7_clean_inv_way;                     /**< Clean and Invalidate by Way, offset: 0x7FC */
  uint8_t RESERVED_9[256];
  uint32_t reg9_d_lockdown0;                       /**< Data Lockdown 0 Register, offset: 0x900 */
  uint32_t reg9_i_lockdown0;                       /**< Instruction Lockdown 0 Register, offset: 0x904 */
  uint32_t reg9_d_lockdown1;                       /**< Cache lockdown by way, offset: 0x908 */
  uint32_t reg9_i_lockdown1;                       /**< Instruction Lockdown 0 Register, offset: 0x90C */
  uint32_t reg9_d_lockdown2;                       /**< Data Lockdown 0 Register, offset: 0x910 */
  uint32_t reg9_i_lockdown2;                       /**< Instruction Lockdown 0 Register, offset: 0x914 */
  uint32_t reg9_d_lockdown3;                       /**< Data Lockdown 3 Register, offset: 0x918 */
  uint32_t reg9_i_lockdown3;                       /**< Instruction Lockdown 3 Register, offset: 0x91C */
  uint32_t reg9_d_lockdown4;                       /**< Data Lockdown 4 Register, offset: 0x920 */
  uint32_t reg9_i_lockdown4;                       /**< Instruction Lockdown 4 Register, offset: 0x924 */
  uint32_t reg9_d_lockdown5;                       /**< Data Lockdown 5 Register, offset: 0x928 */
  uint32_t reg9_i_lockdown5;                       /**< Instruction Lockdown 5 Register, offset: 0x92C */
  uint32_t reg9_d_lockdown6;                       /**< Data Lockdown 6 Register, offset: 0x930 */
  uint32_t reg9_i_lockdown6;                       /**< Instruction Lockdown 6 Register, offset: 0x934 */
  uint32_t reg9_d_lockdown7;                       /**< Data Lockdown 7 Register, offset: 0x938 */
  uint32_t reg9_i_lockdown7;                       /**< Instruction Lockdown 7 Register, offset: 0x93C */
  uint8_t RESERVED_10[16];
  uint32_t reg9_lock_line_en;                      /**< , offset: 0x950 */
  uint32_t reg9_unlock_way;                        /**< , offset: 0x954 */
  uint8_t RESERVED_11[680];
  uint32_t reg12_addr_filtering_start;             /**< Address Filtering Start Register, offset: 0xC00 */
  uint32_t reg12_addr_filtering_end;               /**< , offset: 0xC04 */
  uint8_t RESERVED_12[824];
  uint32_t reg15_debug_ctrl;                       /**< Debug Control Register, offset: 0xF40 */
  uint8_t RESERVED_13[28];
  uint32_t reg15_prefetch_ctrl;                    /**< Prefetch Control Register, offset: 0xF60 */
  uint8_t RESERVED_14[28];
  uint32_t reg15_power_ctrl;                       /**< Power Control Register, offset: 0xF80 */
} volatile * L2C_MemMapPtr;

typedef struct MSCM_MemMap 
{
  vuint32_t CPxTYPE;            /* offset: 0x00000000*/
  vuint32_t CPxNUM;             /* offset: 0x00000004*/
  vuint32_t CPxMASTER;          /* offset: 0x00000008*/
  vuint32_t CPxCOUNT;           /* offset: 0x0000000C*/
  vuint32_t CPxCFG0;            /* offset: 0x00000010*/
  vuint32_t CPxCFG1;            /* offset: 0x00000014*/
  vuint32_t CPxCFG2;            /* offset: 0x00000018*/
  vuint32_t CPxCFG3;            /* offset: 0x0000001C*/
  vuint32_t CP0TYPE;            /* offset: 0x00000020*/
  vuint32_t CP0NUM;             /* offset: 0x00000024*/
  vuint32_t CP0MASTER;          /* offset: 0x00000028*/
  vuint32_t CP0COUNT;           /* offset: 0x0000002C*/
  vuint32_t CP0CFG0;            /* offset: 0x00000030*/
  vuint32_t CP0CFG1;            /* offset: 0x00000034*/
  vuint32_t CP0CFG2;            /* offset: 0x00000038*/
  vuint32_t CP0CFG3;            /* offset: 0x0000003C*/
  vuint32_t CP1TYPE;            /* offset: 0x00000040*/
  vuint32_t CP1NUM;             /* offset: 0x00000044*/
  vuint32_t CP1MASTER;          /* offset: 0x00000048*/
  vuint32_t CP1COUNT;           /* offset: 0x0000004C*/
  vuint32_t CP1CFG0;            /* offset: 0x00000050*/
  vuint32_t CP1CFG1;            /* offset: 0x00000054*/
  vuint32_t CP1CFG2;            /* offset: 0x00000058*/
  vuint32_t CP1CFG3;            /* offset: 0x0000005C*/
  vuint8_t RESERVED0[1952];
  vuint32_t IRCP0IR;            /* offset: 0x00000800*/
  vuint32_t IRCP1IR;            /* offset: 0x00000804*/
  vuint8_t RESERVED1[24];
  vuint32_t IRCPGIR;            /* offset: 0x00000820*/
  vuint8_t RESERVED2[92];
  vuint16_t IRSPRC0;            /* offset: 0x00000880*/
  vuint16_t IRSPRC1;            /* offset: 0x00000882*/
  vuint16_t IRSPRC2;            /* offset: 0x00000884*/
  vuint16_t IRSPRC3;            /* offset: 0x00000886*/
  vuint16_t IRSPRC4;            /* offset: 0x00000888*/
  vuint16_t IRSPRC5;            /* offset: 0x0000088A*/
  vuint16_t IRSPRC6;            /* offset: 0x0000088C*/
  vuint16_t IRSPRC7;            /* offset: 0x0000088E*/
  vuint16_t IRSPRC8;            /* offset: 0x00000890*/
  vuint16_t IRSPRC9;            /* offset: 0x00000892*/
  vuint16_t IRSPRC10;           /* offset: 0x00000894*/
  vuint16_t IRSPRC11;           /* offset: 0x00000896*/
  vuint16_t IRSPRC12;           /* offset: 0x00000898*/
  vuint16_t IRSPRC13;           /* offset: 0x0000089A*/
  vuint16_t IRSPRC14;           /* offset: 0x0000089C*/
  vuint16_t IRSPRC15;           /* offset: 0x0000089E*/
  vuint16_t IRSPRC16;           /* offset: 0x000008A0*/
  vuint16_t IRSPRC17;           /* offset: 0x000008A2*/
  vuint16_t IRSPRC18;           /* offset: 0x000008A4*/
  vuint16_t IRSPRC19;           /* offset: 0x000008A6*/
  vuint16_t IRSPRC20;           /* offset: 0x000008A8*/
  vuint16_t IRSPRC21;           /* offset: 0x000008AA*/
  vuint16_t IRSPRC22;           /* offset: 0x000008AC*/
  vuint16_t IRSPRC23;           /* offset: 0x000008AE*/
  vuint16_t IRSPRC24;           /* offset: 0x000008B0*/
  vuint16_t IRSPRC25;           /* offset: 0x000008B2*/
  vuint16_t IRSPRC26;           /* offset: 0x000008B4*/
  vuint16_t IRSPRC27;           /* offset: 0x000008B6*/
  vuint16_t IRSPRC28;           /* offset: 0x000008B8*/
  vuint16_t IRSPRC29;           /* offset: 0x000008BA*/
  vuint16_t IRSPRC30;           /* offset: 0x000008BC*/
  vuint16_t IRSPRC31;           /* offset: 0x000008BE*/
  vuint16_t IRSPRC32;           /* offset: 0x000008C0*/
  vuint16_t IRSPRC33;           /* offset: 0x000008C2*/
  vuint16_t IRSPRC34;           /* offset: 0x000008C4*/
  vuint16_t IRSPRC35;           /* offset: 0x000008C6*/
  vuint16_t IRSPRC36;           /* offset: 0x000008C8*/
  vuint16_t IRSPRC37;           /* offset: 0x000008CA*/
  vuint16_t IRSPRC38;           /* offset: 0x000008CC*/
  vuint16_t IRSPRC39;           /* offset: 0x000008CE*/
  vuint16_t IRSPRC40;           /* offset: 0x000008D0*/
  vuint16_t IRSPRC41;           /* offset: 0x000008D2*/
  vuint16_t IRSPRC42;           /* offset: 0x000008D4*/
  vuint16_t IRSPRC43;           /* offset: 0x000008D6*/
  vuint16_t IRSPRC44;           /* offset: 0x000008D8*/
  vuint16_t IRSPRC45;           /* offset: 0x000008DA*/
  vuint16_t IRSPRC46;           /* offset: 0x000008DC*/
  vuint16_t IRSPRC47;           /* offset: 0x000008DE*/
  vuint16_t IRSPRC48;           /* offset: 0x000008E0*/
  vuint16_t IRSPRC49;           /* offset: 0x000008E2*/
  vuint16_t IRSPRC50;           /* offset: 0x000008E4*/
  vuint16_t IRSPRC51;           /* offset: 0x000008E6*/
  vuint16_t IRSPRC52;           /* offset: 0x000008E8*/
  vuint16_t IRSPRC53;           /* offset: 0x000008EA*/
  vuint16_t IRSPRC54;           /* offset: 0x000008EC*/
  vuint16_t IRSPRC55;           /* offset: 0x000008EE*/
  vuint16_t IRSPRC56;           /* offset: 0x000008F0*/
  vuint16_t IRSPRC57;           /* offset: 0x000008F2*/
  vuint16_t IRSPRC58;           /* offset: 0x000008F4*/
  vuint16_t IRSPRC59;           /* offset: 0x000008F6*/
  vuint16_t IRSPRC60;           /* offset: 0x000008F8*/
  vuint16_t IRSPRC61;           /* offset: 0x000008FA*/
  vuint16_t IRSPRC62;           /* offset: 0x000008FC*/
  vuint16_t IRSPRC63;           /* offset: 0x000008FE*/
  vuint16_t IRSPRC64;           /* offset: 0x00000900*/
  vuint16_t IRSPRC65;           /* offset: 0x00000902*/
  vuint16_t IRSPRC66;           /* offset: 0x00000904*/
  vuint16_t IRSPRC67;           /* offset: 0x00000906*/
  vuint16_t IRSPRC68;           /* offset: 0x00000908*/
  vuint16_t IRSPRC69;           /* offset: 0x0000090A*/
  vuint16_t IRSPRC70;           /* offset: 0x0000090C*/
  vuint16_t IRSPRC71;           /* offset: 0x0000090E*/
  vuint16_t IRSPRC72;           /* offset: 0x00000910*/
  vuint16_t IRSPRC73;           /* offset: 0x00000912*/
  vuint16_t IRSPRC74;           /* offset: 0x00000914*/
  vuint16_t IRSPRC75;           /* offset: 0x00000916*/
  vuint16_t IRSPRC76;           /* offset: 0x00000918*/
  vuint16_t IRSPRC77;           /* offset: 0x0000091A*/
  vuint16_t IRSPRC78;           /* offset: 0x0000091C*/
  vuint16_t IRSPRC79;           /* offset: 0x0000091E*/
  vuint16_t IRSPRC80;           /* offset: 0x00000920*/
  vuint16_t IRSPRC81;           /* offset: 0x00000922*/
  vuint16_t IRSPRC82;           /* offset: 0x00000924*/
  vuint16_t IRSPRC83;           /* offset: 0x00000926*/
  vuint16_t IRSPRC84;           /* offset: 0x00000928*/
  vuint16_t IRSPRC85;           /* offset: 0x0000092A*/
  vuint16_t IRSPRC86;           /* offset: 0x0000092C*/
  vuint16_t IRSPRC87;           /* offset: 0x0000092E*/
  vuint16_t IRSPRC88;           /* offset: 0x00000930*/
  vuint16_t IRSPRC89;           /* offset: 0x00000932*/
  vuint16_t IRSPRC90;           /* offset: 0x00000934*/
  vuint16_t IRSPRC91;           /* offset: 0x00000936*/
  vuint16_t IRSPRC92;           /* offset: 0x00000938*/
  vuint16_t IRSPRC93;           /* offset: 0x0000093A*/
  vuint16_t IRSPRC94;           /* offset: 0x0000093C*/
  vuint16_t IRSPRC95;           /* offset: 0x0000093E*/
  vuint16_t IRSPRC96;           /* offset: 0x00000940*/
  vuint16_t IRSPRC97;           /* offset: 0x00000942*/
  vuint16_t IRSPRC98;           /* offset: 0x00000944*/
  vuint16_t IRSPRC99;           /* offset: 0x00000946*/
  vuint16_t IRSPRC100;          /* offset: 0x00000948*/
  vuint16_t IRSPRC101;          /* offset: 0x0000094A*/
  vuint16_t IRSPRC102;          /* offset: 0x0000094C*/
  vuint16_t IRSPRC103;          /* offset: 0x0000094E*/
  vuint16_t IRSPRC104;          /* offset: 0x00000950*/
  vuint16_t IRSPRC105;          /* offset: 0x00000952*/
  vuint16_t IRSPRC106;          /* offset: 0x00000954*/
  vuint16_t IRSPRC107;          /* offset: 0x00000956*/
  vuint16_t IRSPRC108;          /* offset: 0x00000958*/
  vuint16_t IRSPRC109;          /* offset: 0x0000095A*/
  vuint16_t IRSPRC110;          /* offset: 0x0000095C*/
  vuint16_t IRSPRC111;          /* offset: 0x0000095E*/
} volatile *MSCM_MemMapPtr;

#define CA5L2C_BASE_PTR                         ((L2C_MemMapPtr)0x40006000u)
#define L2C_reg7_clean_way_REG(base)            ((base)->reg7_clean_way)
#define CA5L2C_reg7_clean_way                   L2C_reg7_clean_way_REG(CA5L2C_BASE_PTR)
#define L2C_reg7_clean_way_Way_MASK             0xFFFFu

#define L2C_reg7_cache_sync_REG(base)           ((base)->reg7_cache_sync)
#define CA5L2C_reg7_cache_sync                  L2C_reg7_cache_sync_REG(CA5L2C_BASE_PTR)
#define L2C_reg7_cache_sync_C_MASK     			0x1u

#define MSCM_BASE_PTR                       	((MSCM_MemMapPtr)0x40001000)
#define MSCM_CPxCFG1_L2SZ_SHIFT                	(24)
#define MSCM_CPxCFG1_L2SZ_MASK                 	((0x000000FF) << (MSCM_CPxCFG1_L2SZ_SHIFT))

/*FUNCTION*-------------------------------------------------------------------
*
* Function Name    : _a5_dcache_flush
* Returned Value   : none
* Comments         :
*   This function FLUSH (CLEAN) all lines of cachce (all sets in all ways)
*   Size of line for A5 L1 cache is 32B.
*
*END*----------------------------------------------------------------------*/

static void _a5_dcache_flush(void)
{
    uint32_t csid, wayset;  /* Cache Size ID */
    int32_t num_sets, num_ways; /* number of sets  */

    MRC(15, 1, csid, 0, 0, 0);    /* Read Cache Size ID */
    /* Fill number of sets  and number of ways from csid register  This walues are decremented by 1*/
    num_ways = (csid >> 0x03) & 0x3FFu;
    while (num_ways >= 0)
    {
        num_sets = (csid >> 0x13) & 0x7FFFu;
        while (num_sets >= 0 )
        {
            wayset = (num_sets << 5u) | (num_ways << 30u);
            /* FLUSH (clean) line if we know set and way */
            MCR(15, 0, wayset, 7, 10, 2);
            num_sets--;
        }
        num_ways--;
    }
    /* All Cache, Branch predictor and TLB maintenance operations before followed instruction complete*/
    DSB();
}

/*FUNCTION*-------------------------------------------------------------------
*
* Function Name    : _l2c310_cache_flush
* Returned Value   : none
* Comments         :
*   This function FLUSH (CLEAN) all lines of cachce (all sets in all ways)
*   Size of line for A5 L2 cache is 32B.
*
*END*----------------------------------------------------------------------*/

static void _l2c310_cache_flush(void)
{
    //read count of ways and clean only them
    CA5L2C_reg7_clean_way |= L2C_reg7_clean_way_Way_MASK;
	
    /* Drain the STB. Operation complete when all buffers, LRB, LFB, STB, and EB, are emptyDrain the STB. Operation complete when all buffers, LRB, LFB, STB, and EB, are empty */
    while (CA5L2C_reg7_cache_sync & L2C_reg7_cache_sync_C_MASK);
}

/*FUNCTION*-------------------------------------------------------------------
*
* Function Name    : _dcache_flush
* Returned Value   : none
* Comments         :
*   This function FLUSH (CLEAN) all lines of cachce (all sets in all ways)
*   Size of line for vybrid L1 cache is 32B.
*
*END*----------------------------------------------------------------------*/

void raw_cache_flush(void)
{
#if PSP_HAS_L1_CACHE
    _a5_dcache_flush();
#endif

#if PSP_HAS_L2_CACHE
    if (MSCM_BASE_PTR->CPxCFG1 & (uint32_t)MSCM_CPxCFG1_L2SZ_MASK)
    {
        _l2c310_cache_flush();
    }
#endif
}